/** @file   www_map.h
 * @brief   Declarations of Map - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:42 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_MAP_H
#define H_WWW_MAP_H

#include "dynamicevent.h"
#include <string>
#include <fstream>
#include "route.h"


namespace WeWantWar {


/** @class  Map
 * @brief   Manages the levelmaps.
 * @author  Tomi Lamminsaari
 *
 * Map-class manages the game levels. Since the Mappy-playback library
 * supports only one active map at a time, there is no need to make
 * instances of this Map-class. All the members are static.
 *
 * The maps consists of two files. First one is the level description file
 * and the other is the map-file itself. The description file contains the
 * information about the mapfile being loaded.
 *
 * The description file contains the information about the weather conditions,
 * doors and dynamic events.
 *
 * @par About the map layers
 * The MappyWin32 supports several map layer. This game uses 3 layers.
 * -Layer 1 : Normal background graphics. Drawn with no transparencies.
 * -Layer 2 : Aliens, bonusitems, lights. See more information below
 * -Layer 3 : Decorative layer. The top most graphics. Drawn with bright
 *            pink being transparent.
 *
 * @par ID codes for the special blocks.
 * The <code>user5</code> - field of the blocks has the ID codes for special
 * tiles. During the level startup process the whole 2nd layer will be searched
 * through and everytime we encounter a tile that has known ID-code in
 * <code>user5</code>-field we create the object in concern. For example tiles
 * that have code <code>16</code> in <code>user5</code>-field are converted
 * to <code> CarnivoreAlien </code>. The location of that tile tells the
 * starting point for that alien.
 * @par
 * The static lights are initialized in same fashion except that the lightblocks
 * should have information about the type of this light in <code>user6</code>-
 * field. The possible values are: 0=streetlight
 */
class Map
{
public:
  
  /** Number of pixels the map has been scrolled horizontally. */
  static int scrollX;
  /** Number of pixels the map has been scrolled vertically. */
  static int scrollY;
  /** The width of the sectors in pixels in current map. */
  static int sectorWidth;
  /** The height of the sectors in pixels in current map. */
  static int sectorHeight;
  /** The width of the current map in pixels */
  static int mwidth;
  /** Height of the current map in pixels */
  static int mheight;
  
  
  /** @struct Block
   * @brief   Represents a single tile in the map.
   *
   * Although the playback library for Mappy Map editor has it's own
   * block-structures, we define new block-structure. This helps
   * us getting rid of polluting our namespace with the global variables
   * defined in there.
   */
  struct Block {
    long data;
    long data2;
    
    unsigned long user1;
    unsigned long user2;
    unsigned short user3;
    unsigned short user4;
    unsigned char user5;
    unsigned char user6;
    unsigned char user7;
    
    unsigned char tl : 1;
    unsigned char tr : 1;
    unsigned char bl : 1;
    unsigned char br : 1;
    unsigned char trigger : 1;
    unsigned char unused1 : 1;
    unsigned char unused2 : 1;
    unsigned char unused3 : 1;
  };
  
  /** We use this block-structure when we return a block data. This same
   * structure is used all over again so every consecutive call of
   * 'blockAt(...)'-method writes new block data over the previus data.
   */
  static Block datablock;
  
  
  /** Are the given coordinates in blocks or in pixels */
  enum DimUnit {
    /** Coordinates should be treated as blocks from topleft corner. */
    IN_BLOCKS,
    /** Coordinates should be treated as pixels from topleft corner. */
    IN_PIXELS
  };
  
  
  /** Enumerated list of supported map layers.
   */
  enum TLayerId {
    /** Normal bacground layer. This is always drawn first. */
    EBackground,
    /** To some elements and objects can be given the starting position by
     * placing certain tiles on this layer.
     */
    EObjects,
    /** Contents of this layer are drawn after the other objects so this
     * is the topmost layer. The tiles should have background transparency
     * setting on.
     */
    EDecorative,
    /** This layer usually contains the mapdata you want to copy over the
     * tiles on @c EBackground - layer.
     */
    EStorage,
    /** The Dynamic Shadows are placed on this layer. */
    EDynamicShadows
  };


  /** Converts the given integer to layer ids
   * @param     aLayerNum         The layer number
   * @return    Matching TLayerId - value
   */
  static TLayerId int2LayerId( int aLayerNum );
  


  ///
  /// General management methods.
  /// ===========================
  
  /** Loads new map. The old map will be destroyed.
   * @param     file              Name of the Map's description file.
   * @return    Nonzero if loading fails. Zero if it's successful.
   */
  static int load(std::string file);
  
  /** Releases the resources allocated for the map
   */
  static void destroy();
  
  
  
  
  /** Selects the active layer. All the map-actions are being targeted to this
   * map layer.
   * @param     layer             Index of the layer.
   * @return    -1 if fails. Otherwise the index of the layer will be returned.
   */
  static int selectLayer( TLayerId layer );
  
  /** Draws part of the map to doublebuffer. The MappyWIn32 supports several
   * map layers. This method draws the background of layer 0.
   * @param     fromX             From this map's X-coordinate.
   * @param     fromY             From this map's Y-coordinate.
   * @param     toX               To this doublebuffer's x-coordinate
   * @param     toY               To this doublebuffer's y-coordinate
   * @param     w                 Width of the area being drawn.
   * @param     h                 Height of the area being drawn.
   * @param     t                 If <code>true</code> the pixels with bright
   *                              ping will be transparent.
   */
  static void redraw( int fromX, int fromY,
                      int toX, int toY,
                      int w, int h, bool t);
  
  /** Draws the foreground of the map to the doublebuffer.
   * @param     fromX             From this map's x-coordinate
   * @param     fromY             From this map's y-coordinate
   * @param     toX               To this doublebuffer's x-coordinate
   * @param     toY               To this doublebuffer's y-coordinate
   * @param     w                 width
   * @param     h                 Height
   */
  static void redrawFG(int fromX, int fromY, int toX, int toY, int w, int h);
  
  /** Updates the map animations.
   */
  static void update();
  
  /** Initializes the map and its animations.
   */
  static void init();



  
  /** Returns the Block at given coordinate.
   * @param     x                 x-coordinate
   * @param     y                 y-coordinate
   * @param     unit              If IN_PIXELS, the (x,y) are treated as pixels
   *                              from topleft corner of the map. If IN_BLOCKS,
   *                              they are treated as blocks from topleft corner.
   */
  static Block* blockAt(int x, int y, DimUnit unit);
  
  /** Replaces the block in map with given new block.
   * @param     x                 X-coordinate
   * @param     y                 Y-coordinate
   * @param     unit              If IN_PIXELS, (x,y) are treated as pixels from
   *                              topleft corner of the map. If IN_BLOCKS, they
   *                              are treated as blocks.
   * @param     data              Number of the new block that will be placed to
   *                              the map.
   */
  static void setBlock( int x, int y, DimUnit unit, int data);
  
  /** Returns the blocknumber that is located at the given position in the map.
   * @param     x                 X coordinate
   * @param     y                 Y coordinate
   * @param     unit              Either @c IN_PIXELS or @c IN_BLOCKS
   * @return    Number of the block at position
   */
  static int blockNumberAt( int x, int y, DimUnit unit );
  
  /** Returns width of the map either in blocks or in pixels.
   * @param     unit              Either IN_BLOCKS or IN_PIXELS
   * @return    Width of the map.
   */
  static int getWidth(DimUnit unit);
  /** Returns the height of the map either in blocks or in pixels.
   * @param     unit              Either IN_BLOCKS or IN_PIXELS
   * @return    Height of the map.
   */
  static int getHeight(DimUnit unit);

  /** Returns the width of the blocks in pixels.
   * @return  Width of the blocks.
   */
  static int getBlockWidth();
  
  /** Returns the height of the blocks in pixels.
   * @return  Height of the blocks.
   */
  static int getBlockHeight();
  
  /** Checks, does the GameObjects collide to the map on given coordinate.
   * @param     x                 X-coordinate
   * @param     y                 Y-coordinate
   * @param     unit              Should the (x,y) be treated as pixels from
   *                              topleft corner or blocks from there.
   * @return    'true' if collisionbit is on.
   */
  static bool collide(int x, int y, DimUnit unit);
  
  /** Checks, does the GameObjects collide to the map on given coordinate
   * (in pixels)
   * @param     p                 Location of the GameObject.
   * @return    'true' if collisionbit is on.
   */
  static bool collide(const eng2d::Vec2D& p);
  
  /** Checks, should the bullet collide to the map on given coordinate.
   * @param     p                 Location of the Bullet.
   * @return    'true' if collides.
   */
  static bool bulletCollide( const eng2d::Vec2D& p );
  
  /** Checks if there is an obstackle between the given points. This method
   * is slow but accurate.
   * @param     p1                Starting point
   * @param     p2                Ending point
   * @return    'true' if there is an obstacle on the way from 'p1' to 'p2'.
   */
  static bool lineBlocked(const eng2d::Vec2D& p1, const eng2d::Vec2D& p2);

  /** Checks, if there is an obstackle between the given points. This method
   * allows you to adjust the accuracy and therefore the efficiency of this
   * method.
   * @param     rP1               Starting point.
   * @param     rP2               Ending point
   * @param     stepSize          Number of pixels we skip over each round as
   *                              we iterate the line from p1 to p2
   * @param     maxDistance       If the distance is between the points is
   *                              greater than this, we assume that there is
   *                              an obstackle.
   * @return    true if there is an obstackle
   */
  static bool lineBlocked( const eng2d::Vec2D& rP1, const eng2d::Vec2D& rP2,
                           float stepSize, float maxDistance );


  ///
  /// Dynamic events
  /// ==============

  /** @struct LevelExitItem
   * @brief   Each level exit is stored in structure like this.
   *
   * A structure that holds the information about the level exits. Levels
   * may have several exits. The level exits are stored in a vector
   * <code> Map::exitList </code>
   */
  struct LevelExitItem {
    /** TopLeft X coordinate in blocks */
    int tlX;
    /** TopLeft Y coordinate in blocks. */
    int tlY;
    /** BottomRight X coordinate in blocks. */
    int brX;
    /** BottomRight Y coordinate in blocks. */
    int brY;
    /** A flag that tells if this exit is already open */
    bool open;
    /** Integer for any purpose. */
    int data1;
    /** Another integer for any purpose. */
    int data2;
  };
  
  /** Bitmask for exit event. Take the return value of Map::update()-method
   * and do bitwise and to find out has the player found level exit.
   */
  static const int EVENT_EXIT = 1;
  
  /** A vector that holds all the dynamic events (spawn, modify, etc. ).
   */
  static std::vector<DynamicEvent*>  eventList;
  
  /** This vector holds the levelexits this level has.
   */
  static std::vector<LevelExitItem*>  exitList;
  
  /** This method reads the information of the EXIT-element.
   * @param     fin               The input stream
   * @return    0 if successful, -1 otherwise.
   */
  static int readExitInformation( std::istream& fin );
  
  
  ///
  /// The methods for waypoint routes
  /// ===============================
  
  /** An empty route we return if requested route cannot be found. */
  static const Route emptyRoute;
  


  /** Class contructor. No need to use this since all the functionality
   * is implemented by static methods.
   */
  Map();
  /** Destructor.
   */
  ~Map();

private:

  /** Private methods **/

  Map(const Map& rO);
  Map& operator = (const Map& rO);

  /** Members **/

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: www_map.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:22+02  lamminsa
 * Initial revision
 *
 */
 
